library(RCurl)

ScopeURLs =
  c(read = 'https://www.googleapis.com/auth/devstorage.read_only',
    write = 'https://www.googleapis.com/auth/devstorage.read_write',
    full = 'https://www.googleapis.com/auth/devstorage.full_control')

getPermission =
function(permission = 'read',
         client_id = getOption("Google.storage.ID"),
         url = 'https://accounts.google.com/o/oauth2/auth', ask = TRUE)
{
  if(grepl('^https', permission))
     scope = permission
  else
     scope = ScopeURLs[permission]

  if(is.na(scope))
    stop("need to specify permission")
    
  args = c(redirect_uri = 'urn:ietf:wg:oauth:2.0:oob',
           scope = as.character(scope),
           client_id = client_id,
           response_type = 'code')
  url = sprintf("%s?%s", url, paste(names(args), args, sep = "=", collapse = '&'))

  browseURL(url)

  if(ask) {
     cat("Cut-and-paste the permission string from your Web browser here: ")
     tmp = readLines(stdin(), 1)
     permissionToken(tmp)
  } else
     invisible(NA)
}

permissionToken =
function(str)
{
     new("OAuth2PermissionToken", str)
}
  


getAuth = getAuthorizationToken =
  #
  # to refresh : bb = getAuth(aa, refresh = TRUE)
  # where aa is an obect of class OAuth2AuthorizationToken
  #
function(token,
         client_id = getOption("Google.storage.ID"),
         client_secret = getOption("Google.storage.Secret"),
         url = 'https://accounts.google.com/o/oauth2/token',
         ..., curl = getCurlHandle(),
         refresh = is(token, "OAuth2AuthorizationToken"))
{
  
   args = c(client_id = client_id,
            client_secret = client_secret)


   if(refresh) {
      args[['grant_type']] = 'refresh_token'
      args[['refresh_token']] = token@refresh_token
   } else {
      args[['grant_type']] = 'authorization_code'
      args[['code']] = token
      args[['redirect_uri']] = 'urn:ietf:wg:oauth:2.0:oob'
   }

   now = Sys.time()
   txt = postForm(url, .params = args, .opts = list(...), style = "POST")

   info = fromJSON(txt)

   if("error" %in% names(info))
       stop("failed to get authorization token: ", info$error)
   
   if(refresh) {
      token@access_token = info$access_token
      token@expiration = now + info$expires_in
      return(token)
   } else 
      makeAuthorizationToken(info, now)
 }


# https://sandbox.google.com/storage/
listBucket =
  #
  # listBucket(auth, "phase1", verbose = TRUE)
  #
  #
function(token, bucket, curl = getCurlHandle(), ..., projectID = getOption('Google.storage.ID'))
{
     # Strip any preceeding /
   if(grepl("^/", bucket))
     bucket = substring(bucket, 2)
   
   url = sprintf("https://commondatastorage.googleapis.com/%s", as.character(bucket))

     # This won't update the original token. For that we need a mutable object.
   token = refreshTokenIfNecessary(token)
   hdr = makeHeader(token, projectID, shortID = TRUE) # If bucket == "/", we want shortID = TRUE.

#   getForm(url, .opts = list(httpheader = hdr))
   ans = getURLContent(url, .opts = list(httpheader = hdr, useragent = "RGoogleStorage", ...))
   parseBucketInfo(ans)
}


download =
  #
  #  z = download(auth, "phase1/test-new-rjson.R", FALSE)
  #
function(token, key, binary = NA,
          curl = getCurlHandle(), ..., projectID = getOption('Google.storage.ID')) 
{
  hdr = makeHeader(token, projectID)
  url = sprintf("https://commondatastorage.googleapis.com/%s", as.character(key))
  getURLContent(url, binary = binary, .opts = list(httpheader = hdr, ...))
}

getACL =
  # Almost the same as download, but we add the acl parameter via a form.
function(token, key, 
          curl = getCurlHandle(), ..., projectID = getOption('Google.storage.ID')) 
{
  hdr = makeHeader(token, projectID)
  url = sprintf("https://commondatastorage.googleapis.com/%s", as.character(key))
  ans = getForm(url, acl = "", .opts = list(httpheader = hdr, ...))
  checkForError(ans)
  parseAccessControlList(ans)
}

setACL =
function(token, key, access, owner = NULL, curl = getCurlHandle(), ...)  
{
  doc = createACLDoc( acl = access, owner = owner)
  txt = saveXML(doc)

  hdr['Content-Type'] = 'application/xml'
  
  url = sprintf("https://commondatastorage.googleapis.com/%s", key)
  rdr = dynCurlReader(curl)
  curlPerform(customrequest = "PUT",
              httpheader = hdr,
              url = url,
              readfunction = charToRaw(txt),
              headerfunction = rdr$update,
              ...,
              curl = curl)

  rdr$value()
}

upload =
  # upload(auth, "phase1/foo", I("This is text"), "text/text")
  # upload(aa, "phase1/humpty", I("This is a string"))
function(token, key, content, type = NA, access = NA,
          curl = getCurlHandle(), ..., projectID = getOption('Google.storage.ID'))  
{
  url = sprintf("https://commondatastorage.googleapis.com/%s", key)

  hdr = makeHeader(token, NA)
  hdr['Date'] = format(Sys.time(), "%B, %d %b %Y %H:%M:%S %Z")
  if(!is.na(access)) 
     hdr['x-goog-acl'] = genericACL(access)

  if(!is.na(type))
    hdr['Content-Type'] = as.character(type)
  
  h = dynCurlReader(curl)
  opts = list(url = url, upload = TRUE,
               httpheader = hdr,
               ...,
               headerfunction = h$update)

  if(is(content, "AsIs") && is.character(content)) {
     content = charToRaw(content)
  }
  
  if(is(content, "raw")) {
     opts$readfunction = content
     hdr['Content-Length'] = length(content)
  } else { # assuming it must be a character
      f = CFILE(content)
      opts$infilesize = file.info(content)[1, "size"]
      opts$readdata = f@ref
  }
  
  curlPerform(.opts = opts, curl = curl)
  
  ans = h$value()
  if(ans == "")
    return(TRUE)

  checkForError(ans)
  xmlParse(ans)
}

makeBucket =
  #
  # makeBucket(aa, "phase5", verbose = TRUE)
  #  Need the NA for the projectID.
  # makeBucket(aa, "phase1/rudolph", verbose = TRUE, projectID = NA)
  #
function(token, key, access = NA,
          curl = getCurlHandle(), ..., projectID = getOption('Google.storage.ID'))
{
#   url = sprintf("https://%s.commondatastorage.googleapis.com", key)
   url = sprintf("https://commondatastorage.googleapis.com/%s", key)
       # Here we have to  modify the project ID to remove the
       #  .apps.googleusercontent.com at the end of the string and leave just the digits.
   hdr = makeHeader(token, projectID, shortID = TRUE)
   hdr['Date'] = format(Sys.time(), "%B, %d %b %Y %H:%M:%S %Z")
   if(!is.na(access)) 
       hdr['x-goog-acl'] = genericACL(access)

   hdr['Content-Length' ] = "0"
   hdr['Expect'] = ""
   hdr['Transfer-Encoding'] = ""   

   h = dynCurlReader(curl)
   curlPerform(url = url, put = TRUE, customrequest = "PUT",
             #  readdata = f@ref, infilesize = 0,
               readfunction = raw(0),
               httpheader = hdr,   ...,
               headerfunction = h$update, curl = curl)
   ans = h$value()
   val = checkForError(ans)
   if(!is.logical(val))
     return(FALSE)
   else if(ans == "")
     TRUE
   else
     ans
}


removeBucket =
  # removeBucket(auth, "phase1/bob")
function(token, bucket,
          curl = getCurlHandle(), ..., projectID = getOption('Google.storage.ID'))  
{
#  h = dynCurlReader(curl, )
  h = basicTextGatherer()
  b = basicTextGatherer()  
  hdr = makeHeader(token, NA)
  url = sprintf("https://commondatastorage.googleapis.com/%s", bucket)
  curlPerform( customrequest = "DELETE",
               url = url,
               httpheader = hdr,
               headerfunction = h$update, writefunction = b$update, ..., curl = curl)
  st = parseHTTPHeader(h$value())
  if(as.integer(st[["status"]]) %/% 100 == 4)
    checkForError(b$value())

  TRUE
}


copy =
function(token, fromKey, toKey, curl = getCurlHandle(), ...)
{
     # Should check fromKey actually exists in Google Storage.

  url = sprintf("https://commondatastorage.googleapis.com/%s", toKey)
  hdr = makeHeader(token, NA, TRUE)
  hdr['x-goog-copy-source'] = fromKey
  hdr['x-goog-metadata-directive'] = "REPLACE"
  hdr['Content-Length'] =  "0"

  rdr = dynCurlReader(curl)
  args = list(httpheader = hdr,
              url = url,
              headerfunction = rdr$update,
              customrequest = "PUT",
              ...
             )

  curlPerform(.opts = args, curl = curl)
  checkForError(rdr$value())
  doc = xmlParse(rdr$value())
  xmlToList(doc)
}
