\name{getAuth}
\alias{getAuth}
\alias{getAuthorizationToken}
\alias{getPermission}
\alias{refreshToken}
\title{OAuth2 functions for Google Storage}
\description{
  These functions establish the permissions and authorization tokens
  with which we can interact with Google Storage and access the data and buckets
  for a particular project.
  The typical sequence is to call
  \code{getPermission} and use the Web browser to grant the permissions
  and obtain a token. This is copied back into R and then we have an
  \code{OAuth2PermissionToken} object.
  Next, we obtain an authorization token with which we can make
  authenticated requests. This is done via \code{getAuth}.
  No user interaction or credentials (login and passwword) are needed
  for this step.
  Then this  \code{OAuth2AuthorizationToken} is used in all functions
  that interact with Google Storage and access the data and storage
  there.
  The \code{refreshToken} token is used when the access/authorization
  token
  expires. We use the object returned from \code{getAuth} to update
  the access token by using its refresh token.

  Some of the functions automatically refresh the token before making
  a request. However, they do not update the token object and so
  must do this each time. It is more efficient to update the
  token object manually every hour and then use in subsequent calls
  rather than having each call request a new token.
  In the future, we will probably implement a mutable object that updates the
  token automatically in between calls an stores it across calls.
}
\usage{
getPermission(permission = 'read',
               client_id = getOption("Google.storage.ID"),
                url = 'https://accounts.google.com/o/oauth2/auth',
               ask = TRUE)
getAuth(token,
         client_id = getOption("Google.storage.ID"),
         client_secret = getOption("Google.storage.Secret"),
         url = "https://accounts.google.com/o/oauth2/token",
         ..., curl = getCurlHandle(),
         refresh =  is(token, "OAuth2AuthorizationToken"))
refreshToken(auth, now = Sys.time(), eps = 5, ...)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{permission}{a string identifying the permissions being
    requested. These are 'read', 'write' and 'full'.}  
  \item{ask}{a logical value that controls whether
  \code{getPermission} waits for the caller to interact with the Web
  browser and then cut-and-paste the token returned by Google.
  This then takes care of constructing the "enriched" R object
  representing the token.}
  \item{token}{the original permission token from
    \code{getPermission} that is obtained by the user logging into
    Google and granting the requested permission by our application.}
  \item{auth}{the OAuth2AuthorizationToken object original obtained from
  a call to \code{getAuth}. This is then updated by requesting a "fresh"
   access token.}
  \item{client_id,client_secret}{the identity and secret of the
    application which is attempting to connect to the Google Storage to
    access. It is best to specify this as hidden variables such as in a
    closure or  as options, but not hard-coded in publically available code.
  }
  \item{url}{the URL to which the request should ebe made. This is
    rarely specified by the caller.}
  \item{curl}{an optional existing curl handle that is used for the HTTP
  requests in this operation}
  \item{\dots}{additional parameters that are passed to \code{\link[RCurl]{curlPerform}}}  
  \item{refresh}{a logical value that controls whether this operation is
   to refresh an existing authorization token or to establish an initial
 }

 \item{now}{the current time which is used to compare the expiration
 time of the authorization token and to determine if we need to renew it.}
 \item{eps}{the number of seconds to subtract when comparing the current
 time to the expiration date. This is basically a fudge factor we build
 in to add the time it will take us to make the request and so the
 possibility that the authorization token will expire during the time we
 are preparing and sending the request.}
}
\value{
  \code{getPermission} with \code{ask = TRUE} returns an object of class
  \code{"OAuth2PermissionToken"}.
  With \code{ask = FALSE}, this just returns a value of \code{NULL}
  and the user is expected to create the token from the string Google
  displays in the Web page.

  \code{getAuth} returns an object of class
  \code{OAuth2AuthorizationToken}.
  This is then passed to the other functions that interact with the
  Google Storage API.

  \code{refreshToken} takes and updates an
  \code{OAuth2AuthorizationToken} object.
}
\references{
  \url{http://code.google.com/apis/storage/docs/accesscontrol.html}
}
\author{
Duncan Temple Lang
}
\seealso{
  All of the high-level functions of the package
  to create and list buckets, upload and download
  content and get and set access control lists.
}
\examples{
  \dontrun{
   options(Google.storage.ID = '123457891011.apps.googleusercontent.com',
            Google.storage.Secret = 'itsASecret')
   token = getPermission('full')
       # cut and paste the string from the Web browser into R
   aa = getAuth(token)

   listBucket(aa, "")
       
  }
}
\keyword{programming}
\keyword{data}
